# README:
# This function creates random patterns of equal initial vegetation cover 
# that can be tuned using a number or parameters described below.
#
# Target initial vegetation cover is always 10% of the entire grid
# 'Patch number' (integer): specifies the desired number of distinct vegetated patches (composed of multiple vegetated cells)
# 'Matrix length' (integer): specifies the length of the square matrix that will be outputed (i.e., matrixLength = 10 will output a 10 X 10 matrix)
# 'Patch size variance' (numeric): varies the number of vegetated cells composing each patch. A larger number will produce greater varaince.
# 'Gravity' (positive or negative integer): 
#             Positive values cause patches to attract each other in initial setup, causing over-dispersed patterns 
#             Negative values cause patches to repulse each other in initial setup, causing regular patterns
#             '0' produces unbiased random 'Poisson' patterns
#             The larger the value, the more over-dispersion or regularization. The effect saturates at high values.
#
# Dependencies: 'gravityPatch.R', 'matrixToPoints.R', 'pointsToMatrix.R'

initialPatchMaker = function(patchNumber, matrixLength, patchSizeVariance, gravity){

  # Using gravity points to change point clustering
  #set.seed(1)
  Mi = gravityPatch(n_points = patchNumber, 
                    gravityPolarity = sign(gravity), 
                    n_runs = abs(gravity), 
                    matrixLength = matrixLength)
  
  # Vary patch sizes
  n_patches = sum(Mi)
  targetPatchSize = rlnorm(n = n_patches, sdlog = patchSizeVariance)
  targetPatchSize = round(targetPatchSize*(0.1*matrixLength^2/sum(targetPatchSize)))
  targetPatchSize[targetPatchSize < 1] = 1
  
  # Grow patches up diffusively to desired size.  Add some noise to make them unique. 
  # (This part is slow)
  i_patch = 1
  points = which(Mi == 1, arr.ind = T)
  for(i_patch in 1:n_patches){
    
    M = array(0, dim = c(matrixLength,matrixLength))
    x = as.numeric(points[i_patch,1])
    y = as.numeric(points[i_patch,2])
    M[x,y] = 1
    while(sum(round(M)) < targetPatchSize[i_patch]){
      roll = rollMatrix(M, n_directions = 4, operation = 'sum')
      M = M + roll*0.01
      M[M > 1] = 1
    }
    M[M < 0.5] = 0
    
    Mout = array(0, dim = c(matrixLength,matrixLength))
    Mout[M >= 0.9] = 1
    x = rep(1:matrixLength, times = matrixLength)
    y = rep(1:matrixLength, each = matrixLength)
    index = 1:length(x)
    probabilityWeight = as.vector(M)
    probabilityWeight[probabilityWeight < 0.5] = 0
    while(sum(round(Mout)) < targetPatchSize[i_patch] & sum(probabilityWeight) > 0){
      select = sample(index, prob = probabilityWeight, size = 1)
      if(Mi[x[select],y[select]] == 0){
        Mout[x[select],y[select]] = 1
      }
      probabilityWeight = probabilityWeight[index != select]
      index = index[index != select]
    }
    #print(paste(i_patch,'/',n_patches))
    Mi = Mi + Mout
  }
  Mi[Mi > 1] = 1
  
  # Reduce the patch coverage to target coverage value
  if(sum(Mi) > 0.1*matrixLength^2){
    x = rep(1:matrixLength, times = matrixLength)
    y = rep(1:matrixLength, each = matrixLength)
    index = 1:length(x)
    edgeM = rollMatrix(Mi, n_directions = 4, operation = 'sum')
    edgeM = 1 - edgeM/max(edgeM)
    edgeM[Mi == 0] = 0
    
    probabilityWeight = as.vector(edgeM)
    select = sample(index, prob = probabilityWeight, size = sum(Mi) - 0.1*matrixLength^2)
    for(i in 1:length(select)){
      Mi[x[select[i]],y[select[i]]] = 0
    }
  }
  
  # Increase the patch covereage to target coverage value
  if(sum(Mi) < 0.1*matrixLength^2){
    x = rep(1:matrixLength, times = matrixLength)
    y = rep(1:matrixLength, each = matrixLength)
    index = 1:length(x)
    edgeM = sign(rollMatrix(Mi, n_directions = 4, operation = 'sum'))
    edgeM[Mi == 1] = 0
    
    
    if(sum(edgeM) > sum(Mi) - 0.1*matrixLength^2){
      probabilityWeight = as.vector(edgeM)
      select = sample(index, prob = probabilityWeight, size = 0.1*matrixLength^2 - sum(Mi))
      for(i in 1:length(select)){
        Mi[x[select[i]],y[select[i]]] = 1
      }
    } else {
      select = sample(index, size = 0.1*matrixLength^2 - sum(Mi))
      for(i in 1:length(select)){
        Mi[x[select[i]],y[select[i]]] = 1
      }
    }
  }
  return(Mi)
}

